<?php
/**
 * My Training – Enrolled courses list with Moodle default table + pagination
 * Status column coloured by state (PHP-7 compatible).
 */
require_once(__DIR__ . '/../../config.php');
require_login();

$PAGE->set_url(new moodle_url('/local/custompage/training.php'));
$PAGE->set_context(context_system::instance());
$PAGE->set_pagelayout('standard');
$PAGE->set_title('My Training');
$PAGE->set_heading('My Training');

/* ---------- Custom CSS: centre everything & compact rows ---------- */
$customcss = <<<CSS
.training-wrapper { 
    overflow-x:auto; 
    -webkit-overflow-scrolling:touch;
    text-align:center;                /* centre the wrapper */
}

/* Centre all header/data cells horizontally & vertically */
#trainingtbl th,
#trainingtbl td {
   
    vertical-align: middle;
    padding-top: 6px;                 /* smaller vertical padding */
    padding-bottom: 6px;
}

/* Optional: slightly smaller font for a compact look */
#trainingtbl {
    font-size: 14px;
    margin-left: auto;                /* centre the table itself */
    margin-right: auto;
}

/* status colours */
.status-started    { color:#0d6efd; font-weight:bold; }  /* blue */
.status-progress   { color:#fd7e14; font-weight:bold; }  /* orange */
.status-completed  { color:#198754; font-weight:bold; }  /* green */
.status-default    { color:#6c757d; font-weight:bold; }  /* grey */
CSS;

echo $OUTPUT->header();
echo html_writer::tag('style', $customcss);
echo html_writer::tag('h3', 'My Training', ['class' => 'mb-3 text']);

/* -------- Helper: format seconds -> "0h 0m 1s" -------------------- */
function format_duration($seconds): string {
    if (empty($seconds) || !is_numeric($seconds)) {
        return '0h 0m 0s';
    }
    $h = floor($seconds / 3600);
    $m = floor(($seconds % 3600) / 60);
    $s = $seconds % 60;
    return sprintf('%dh %dm %ds', $h, $m, $s);
}

/* ---------------- Pagination setup -------------------------------- */
$perpage = 10;
$page    = optional_param('page', 0, PARAM_INT);

$courses      = enrol_get_my_courses(['id','fullname'], 'fullname ASC');
$totalcount   = count($courses);
$courses_page = array_slice($courses, $page * $perpage, $perpage);

/* ---------------- Build table ------------------------------------- */
$table = new html_table();
$table->id = 'trainingtbl';
$table->attributes['class'] = 'generaltable';
$table->align = ['left','center','center','center']; // all columns centred
$table->head  = ['Course','Duration','Status','Action'];

foreach ($courses_page as $course) {
    // Duration from custom fields
    $duration_value = $DB->get_field_sql("
        SELECT d.value
          FROM {customfield_data} d
          JOIN {customfield_field} f ON f.id = d.fieldid
         WHERE d.instanceid = :cid AND f.shortname = 'course_duration_value'",
        ['cid' => $course->id]);

    $duration_unit = $DB->get_field_sql("
        SELECT d.value
          FROM {customfield_data} d
          JOIN {customfield_field} f ON f.id = d.fieldid
         WHERE d.instanceid = :cid AND f.shortname = 'duration'",
        ['cid' => $course->id]);

    $seconds = 0;
    if (is_numeric($duration_value)) {
        switch (strtoupper($duration_unit)) {
            case 'SECONDS': $seconds = $duration_value; break;
            case 'MINUTES': $seconds = $duration_value * 60; break;
            case 'HOURS'  : $seconds = $duration_value * 3600; break;
            default       : $seconds = $duration_value;
        }
    }
    $duration = format_duration($seconds);

    // Latest status
    $statusraw = $DB->get_field_sql("
        SELECT lp.activityprogress
          FROM {local_ltiprogress} lp
         WHERE lp.userid = :uid AND lp.ltiid = :cid
      ORDER BY lp.timemodified DESC, lp.timecreated DESC
         LIMIT 1",
        ['uid' => $USER->id, 'cid' => $course->id]) ?? 'Not Started';

    $label = strtolower($statusraw);
    if (strpos($label, 'complete') !== false) {
        $statusclass = 'status-completed';
        $status = 'Completed';
    } elseif (strpos($label, 'progress') !== false) {
        $statusclass = 'status-progress';
        $status = 'In Progress';
    } elseif (strpos($label, 'start') !== false) {
        $statusclass = 'status-started';
        $status = 'Started';
    } else {
        $statusclass = 'status-default';
        $status = 'Not Started';
    }
    $statushtml = html_writer::span($status, $statusclass);

    // Access Now button
    $accessbtn = html_writer::link(
        new moodle_url('/course/view.php', ['id' => $course->id]),
        'Access Now',
        ['class' => 'btn btn-primary btn-sm']
    );

    $table->data[] = [
        format_string($course->fullname),
        $duration,
        $statushtml,
        $accessbtn
    ];
}

/* ---------------- Output table + pagination ----------------------- */
echo html_writer::start_div('training-wrapper');
echo html_writer::table($table);
echo html_writer::end_div();

echo $OUTPUT->paging_bar($totalcount, $page, $perpage,
    new moodle_url('/local/custompage/training.php'));

echo $OUTPUT->footer();
