<?php
require_once(__DIR__ . '/../../config.php');
require_login();
require_capability('moodle/site:config', context_system::instance()); // Only managers/admins

global $DB, $OUTPUT, $PAGE;

// Set up page
$PAGE->set_url(new moodle_url('/local/custompage/manage_req.php'));
$PAGE->set_context(context_system::instance());
$PAGE->set_title('Manage Course Requests');
$PAGE->set_heading('Manage Course Requests');

// Include jQuery, DataTables & SweetAlert2
echo html_writer::tag('link', '', [
    'rel' => 'stylesheet',
    'type' => 'text/css',
    'href' => 'https://cdn.datatables.net/1.13.6/css/jquery.dataTables.min.css'
]);
echo html_writer::tag('script', '', [
    'src' => 'https://code.jquery.com/jquery-3.6.0.min.js'
]);
echo html_writer::tag('script', '', [
    'src' => 'https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js'
]);
echo html_writer::tag('script', '', [
    'src' => 'https://cdn.jsdelivr.net/npm/sweetalert2@11'
]);

// Handle enrol button click
if (optional_param('enrol', 0, PARAM_INT)) {
    $requestid = required_param('id', PARAM_INT);

    // Get request info
    $request = $DB->get_record('course_enrol_requests', ['id' => $requestid]);
    if ($request) {
        // Enrol user using manual enrolment plugin
        $enrol = enrol_get_plugin('manual');
        if ($enrol) {
            $instances = enrol_get_instances($request->courseid, true);
            foreach ($instances as $instance) {
                if ($instance->enrol == 'manual') {
                    $enrol->enrol_user($instance, $request->userid);
                }
            }
        }

        // Delete request after enrolment
        $DB->delete_records('course_enrol_requests', ['id' => $requestid]);
    }

    redirect(new moodle_url('/local/custompage/manage_req.php'));
}

// Fetch all pending enrol requests
$requests = $DB->get_records_sql("
    SELECT r.id, r.courseid, r.userid, r.requested_at, 
           c.fullname AS coursename, u.firstname, u.lastname
    FROM {course_enrol_requests} r
    JOIN {course} c ON r.courseid = c.id
    JOIN {user} u ON r.userid = u.id
    ORDER BY r.requested_at DESC
");

echo $OUTPUT->header();
echo html_writer::tag('h3', 'Pending Course Enrol Requests', ['class' => 'mb-3']);

// Create responsive table
$table = new html_table();
$table->id = 'managereq';
$table->attributes['class'] = 'table table-striped table-bordered table-hover';
$table->head = ['Course Name', 'Requested By', 'Requested Date', 'Action'];

foreach ($requests as $req) {
    $userfullname = fullname($req);
    $enrolurl = new moodle_url('/local/custompage/manage_req.php', ['enrol' => 1, 'id' => $req->id]);

    // Button without default confirm box, handled via SweetAlert
    $enrolbtn = html_writer::link($enrolurl, 'Enrol', [
        'class' => 'btn btn-primary btn-sm enrol-btn',
        'data-url' => $enrolurl
    ]);

    // Show only date (no time)
    $table->data[] = [
        format_string($req->coursename),
        $userfullname,
        userdate($req->requested_at, get_string('strftimedate', 'langconfig')),
        $enrolbtn
    ];
}

echo html_writer::table($table);
?>

<!-- DataTables + SweetAlert2 Script -->
<script>
jQuery(document).ready(function($) {
    // Initialize DataTable
    $('#managereq').DataTable({
        "pageLength": 10,
        "lengthMenu": [[10, 25, 50, 100], [10, 25, 50, 100]],
        "order": [],
        "responsive": true,
        "language": {
            "search": "Search:",
            "lengthMenu": "Show _MENU_ entries",
            "info": "Showing _START_ to _END_ of _TOTAL_ requests",
            "paginate": {
                "first": "First",
                "last": "Last",
                "next": "Next",
                "previous": "Previous"
            }
        }
    });

    // SweetAlert2 Confirmation for Enrol
    $(document).on('click', '.enrol-btn', function(e) {
        e.preventDefault();
        var url = $(this).data('url');

        Swal.fire({
            title: "Are you sure?",
            text: "Assign the course!",
            icon: "warning",
            showCancelButton: true,
            confirmButtonColor: "#3085d6",
            cancelButtonColor: "#d33",
            cancelButtonText: "No, cancel!",
            confirmButtonText: "Yes, enrol!"
        }).then((result) => {
            if (result.isConfirmed) {
                window.location.href = url;
            }
        });
    });
});
</script>

<?php
echo $OUTPUT->footer();
?>
