<?php
require_once(__DIR__ . '/../config.php');
require_login();
global $DB, $CFG, $OUTPUT, $USER;

$PAGE->set_context(context_system::instance());
$PAGE->set_url(new moodle_url('/course/wcompany.php'));
$PAGE->set_title('Add/Edit Company');

// Debugging
@error_reporting(E_ALL | E_STRICT);
@ini_set('display_errors', '1');
$CFG->debug = (E_ALL | E_STRICT);
$CFG->debugdisplay = 1;

// Get company ID if editing
$companyid = optional_param('id', 0, PARAM_INT);
$company = null;

if ($companyid) {
    $company = $DB->get_record('company', ['id' => $companyid], '*', MUST_EXIST);
}

/* ROLE CHECK */
$manager_role = $DB->get_record('role', ['shortname' => 'manager'], '*', MUST_EXIST);
$ismanager = user_has_role_assignment($USER->id, $manager_role->id);

/* Fetch MANAGERS for admin */
$managers = $DB->get_records_sql("
    SELECT DISTINCT u.id AS userid, u.firstname, u.lastname
    FROM {role_assignments} ra
    JOIN {user} u ON u.id = ra.userid
    WHERE ra.roleid = :roleid
    ORDER BY u.firstname ASC
", ['roleid' => $manager_role->id]);

/* ------------------ FORM SUBMISSION ------------------ */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $record = new stdClass();
        $record->com_name  = required_param('com_name', PARAM_TEXT);
        $record->phone     = optional_param('phone', '', PARAM_RAW_TRIMMED);
        $record->website   = optional_param('website', '', PARAM_RAW_TRIMMED);
        $record->address   = optional_param('address', '', PARAM_TEXT);
        $record->zip       = optional_param('zip', '', PARAM_RAW_TRIMMED);

        /* ✅ NEW EMAIL FIELD */
        $record->email = optional_param('email', '', PARAM_RAW_TRIMMED);

        // Assign manager
        if ($ismanager) {
            $record->class_manager = $USER->id; // manager = logged-in user
        } else {
            $record->class_manager = required_param('class_manager', PARAM_INT);
        }

        if ($companyid) {
            $record->id = $companyid;
            $DB->update_record('company', $record);
            redirect(new moodle_url('/course/wcompanylist.php'), '✅ Company updated successfully!', 2);
        } else {
            $record->timecreated = time();
            $DB->insert_record('company', $record);
            redirect(new moodle_url('/course/wcompany.php'), '✅ Company added successfully!', 2);
        }

    } catch (Exception $e) {
        echo $OUTPUT->notification("❌ Database error: " . $e->getMessage(), 'notifyproblem');
    }
}

echo $OUTPUT->header();
?>
<!-- ------------------ CSS ------------------ -->
<style>
    #page-header { margin-top: 0px !important; }
    #region-main { padding: 0px 0px !important; }

    .company-form-container {
        width: 100%;
        max-width: 1000px;
        margin: 20px auto;
        background: #fff;
        border-radius: 10px;
        box-shadow: 0 4px 20px rgba(0,0,0,0.1);
        font-family: "Segoe UI", sans-serif;
        padding: 10px;
    }

    .company-form-header {
        background-color: #f7f7f7;
        padding: 20px 30px;
        font-size: 22px;
        font-weight: bold;
        color: #2c3e50;
        border-bottom: 1px solid #ddd;
    }

    .company-form-body {
        padding: 30px;
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 25px;
    }

    .form-group label {
        font-weight: 600;
        margin-bottom: 6px;
    }

    .form-group input,
    .form-group select,
    .form-group textarea {
        padding: 12px;
        border: 1px solid #ccc;
        border-radius: 6px;
        font-size: 15px;
        width: 100%;
    }

    textarea {
        resize: vertical;
        height: 100px;
    }

    .company-form-footer {
        grid-column: 1 / -1;
        text-align: right;
    }
</style>

<!-- ------------------ FORM HTML ------------------ -->
<div class="company-form-container">

    <div class="company-form-header">
        <span><?= $companyid ? 'Edit Company' : 'Add Company' ?></span>
        <a href="#" onclick="window.history.back();" class="btn btn-secondary">Back</a>
    </div>

    <form method="post" class="company-form-body">

        <div class="form-group">
            <label>Company Name *</label>
            <input type="text" name="com_name" required
                   value="<?= $company->com_name ?? '' ?>">
        </div>

        <?php if (!$ismanager): ?>
        <div class="form-group">
            <label>Admin *</label>
            <select name="class_manager" required>
                <option value="">Select Admin</option>
                <?php foreach ($managers as $manager): ?>
                    <option value="<?= $manager->userid ?>"
                        <?= isset($company) && $company->class_manager == $manager->userid ? 'selected' : '' ?>>
                        <?= fullname($manager) ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>
        <?php endif; ?>

        <!-- ✅ NEW EMAIL FIELD -->
        <div class="form-group">
            <label>Email</label>
            <input type="email" name="email"
                   value="<?= $company->email ?? '' ?>">
        </div>

        <div class="form-group">
            <label>Phone</label>
            <input type="text" name="phone" value="<?= $company->phone ?? '' ?>">
        </div>

        <div class="form-group">
            <label>Website</label>
            <input type="text" name="website" value="<?= $company->website ?? '' ?>">
        </div>

        <div class="form-group" style="grid-column: 1 / -1;">
            <label>Address</label>
            <textarea name="address"><?= $company->address ?? '' ?></textarea>
        </div>

        <div class="form-group">
            <label>Zip Code</label>
            <input type="text" name="zip" value="<?= $company->zip ?? '' ?>">
        </div>

        <div class="company-form-footer">
            <button type="submit" class="btn btn-primary">
                <?= $companyid ? 'Update' : 'Create' ?>
            </button>
        </div>

    </form>

</div>

<?php echo $OUTPUT->footer(); ?>
