<?php
// File: local/learningpath/learner_paths.php
require_once('../../config.php');
require_login();

global $DB, $USER, $PAGE, $OUTPUT, $CFG;

// restrict to learners only (shortnames commonly used)
function is_user_learner($userid = null) {
    global $DB;
    if (!$userid) global $USER;
    $roles = get_user_roles(context_system::instance(), $userid ? $userid : $USER->id, true);
    foreach ($roles as $r) {
        if (in_array($r->shortname, ['student','learner','participant','studentmember'])) {
            return true;
        }
    }
    return false;
}

if (!is_user_learner($USER->id)) {
    // show standard header and friendly message
    $PAGE->set_url(new moodle_url('/local/learningpath/learner_paths.php'));
    $PAGE->set_context(context_system::instance());
    echo $OUTPUT->header();
    echo $OUTPUT->notification('Access denied — this page is for learners only.', 'notifyproblem');
    echo $OUTPUT->footer();
    exit;
}

$PAGE->set_url(new moodle_url('/local/learningpath/learner_paths.php'));
$PAGE->set_context(context_system::instance());
$PAGE->set_title('My Learning Paths');
$PAGE->set_heading('My Learning Paths');

// include custom css (file in same folder)
$cssfile = new moodle_url('/local/learningpath/learner_styles.css');
$PAGE->requires->css($cssfile);

echo $OUTPUT->header();
echo '<div class="lp-container">';

// get learning paths (show published only)
$learningpaths = $DB->get_records('learning_paths', ['status' => 'Publish'], 'timecreated DESC');

echo '<h2 style="margin-bottom:12px;">My Learning Paths</h2>';

if (empty($learningpaths)) {
    echo '<div class="card p-3">No learning paths available.</div>';
    echo $OUTPUT->footer();
    exit;
}

echo '<div class="lp-grid">';

foreach ($learningpaths as $lp) {
    // total courses in this learning path
    $totalcourses = $DB->count_records_sql("
        SELECT COUNT(lpc.id)
        FROM {learning_path_courses} lpc
        JOIN {learning_path_sections} s ON s.id = lpc.sectionid
        WHERE s.learningpathid = :lid
    ", ['lid' => $lp->id]);

    // completed courses for current user (mdl_course_completions.timecompleted > 0)
    if ($totalcourses > 0) {
        $completedcourses = $DB->count_records_sql("
            SELECT COUNT(cc.id)
            FROM {course_completions} cc
            JOIN {learning_path_courses} lpc ON lpc.courseid = cc.course
            JOIN {learning_path_sections} s ON s.id = lpc.sectionid
            WHERE s.learningpathid = :lid AND cc.userid = :uid AND cc.timecompleted > 0
        ", ['lid' => $lp->id, 'uid' => $USER->id]);
    } else {
        $completedcourses = 0;
    }

    $progresspct = ($totalcourses > 0) ? intval(($completedcourses / $totalcourses) * 100) : 0;

    // find first unlocked course (first section unlocked -> first course; see sequencing rules: first section always unlocked)
    // For overview, we will link Access Now to view page
    $viewurl = new moodle_url('/local/learningpath/learner_path_view.php', ['id' => $lp->id]);

    echo '<div class="lp-card">';
    echo '<div>';
    echo '<div class="lp-title">' . format_string($lp->name) . '</div>';
    echo '<div class="lp-meta">' . $DB->count_records('learning_path_sections', ['learningpathid' => $lp->id]) . ' sections · ' . $totalcourses . ' courses</div>';
    echo '<div class="lp-desc">' . ($lp->description ? format_text($lp->description, FORMAT_HTML) : '<em>No description</em>') . '</div>';

    // progress bar
    echo '<div class="lp-progress-wrap">';
    echo '<div style="display:flex;justify-content:space-between;align-items:center;margin-bottom:6px;">';
    echo '<small>Progress</small><small>' . $progresspct . '%</small>';
    echo '</div>';
    echo '<div class="lp-progress"><i style="width:' . $progresspct . '%;"></i></div>';
    echo '</div>';

    echo '</div>'; // end top

    echo '<div style="display:flex;justify-content:space-between;align-items:center;margin-top:12px;">';
    echo '<a class="btn-secondary" href="' . $viewurl->out(false) . '">Details</a>';
    echo '<a class="btn-primary-full" href="' . $viewurl->out(false) . '">Access Now</a>';
    echo '</div>';

    echo '</div>'; // lp-card
}

echo '</div>'; // grid
echo '</div>'; // container
echo $OUTPUT->footer();
exit;
