<?php
// local/custompage/bulk_upload.php
require_once('../../config.php');
require_login();
require_capability('moodle/user:create', context_system::instance());

global $DB, $USER, $CFG, $OUTPUT, $PAGE;

$PAGE->set_url(new moodle_url('/local/custompage/bulk_upload.php'));
$PAGE->set_context(context_system::instance());
$PAGE->set_title('Bulk User Upload');
$PAGE->set_heading('Bulk User Upload');

require_once($CFG->libdir.'/formslib.php');
require_once($CFG->dirroot.'/user/lib.php');

class bulk_upload_form extends moodleform {
    public function definition() {
        global $DB, $USER;
        $mform = $this->_form;

        // Role select
        $allroles = $DB->get_records('role');
        $roleoptions = [];
        if (in_array($USER->id, [1,2])) {
            foreach ($allroles as $role) {
                if (strtolower($role->shortname) === 'manager') {
                    $roleoptions[$role->id] = $role->name;
                }
            }
        } else {
            foreach ($allroles as $role) {
                if (strtolower($role->shortname) === 'student') {
                    $roleoptions[$role->id] = ucfirst($role->shortname);
                }
            }
        }
        $mform->addElement('select', 'assignrole', 'Assign Role', $roleoptions);
        $mform->setType('assignrole', PARAM_INT);
        $mform->addRule('assignrole', null, 'required');

        // Example CSV
        $url = new moodle_url('/admin/tool/uploaduser/example.csv');
        $mform->addElement('static', 'examplecsv', 'Example CSV', html_writer::link($url, 'example.csv'));

        // CSV file picker
        $mform->addElement('filepicker', 'userfile', 'Upload CSV', null, ['accepted_types' => ['.csv']]);
       // $mform->addRule('userfile', null, 'required');

        // Buttons
        $buttonarray = [];
        $buttonarray[] = $mform->createElement('submit', 'uploaduser', 'Upload User');
        $buttonarray[] = $mform->createElement('submit', 'cancel', 'Cancel');
        $mform->addGroup($buttonarray, 'buttonar', '', [' '], false);

        $mform->addElement('html', '<style>
            #fgroup_buttonar { display: flex; justify-content: space-between; margin-top: 10px; }
            .preview-table { width: 100%; border-collapse: collapse; margin-top:20px; }
            .preview-table th, .preview-table td { border: 1px solid #ddd; padding: 8px; text-align: left; }
            .preview-table th { background-color: #f2f2f2; }
            .notifysuccess { color: green; }
            .error { color: red; }
        </style>');
    }
}

$form = new bulk_upload_form();
echo $OUTPUT->header();

// Handle cancel
if ($form->is_cancelled() || optional_param('cancel', 0, PARAM_BOOL)) {
    redirect(new moodle_url('/local/custompage/bulk_upload.php'));
}

$data = $form->get_data();
$preview = [];

if ($data && isset($data->uploaduser)) {
    $draftitemid = $data->userfile;
    $fs = get_file_storage();
    $usercontext = context_user::instance($USER->id);
    $files = $fs->get_area_files($usercontext->id, 'user', 'draft', $draftitemid, 'id ASC', false);

    if ($files) {
        $file = reset($files);
        $csvcontent = $file->get_content();
        $rows = explode("\n", $csvcontent);

        foreach ($rows as $key => $line) {
            $line = trim($line);
            if ($line === '') continue;

            $cols = str_getcsv($line);
            if ($key == 0) continue; // skip header

            $username  = trim($cols[0] ?? '');
            $firstname = trim($cols[1] ?? '');
            $lastname  = trim($cols[2] ?? '');
            $cohorts   = array_slice($cols, 3); // remaining columns as cohorts
            $status    = '';

            if (empty($username)) {
                $status = 'Username missing';
            } elseif ($DB->record_exists('user', ['username' => $username])) {
                $status = 'User already exists';
            } else {
                $user = new stdClass();
                $user->username   = $username;
                $user->email      = $username; // email = username
                $user->firstname  = $firstname;
                $user->lastname   = $lastname;
                $user->auth       = 'manual';
                $user->mnethostid = $CFG->mnet_localhost_id;
                $user->confirmed  = 1;
               
                try {
                    $userid = user_create_user($user, true, false);
                    role_assign($data->assignrole, $userid, context_system::instance());
                    $status = 'User created';

                    // Handle cohorts
                    foreach ($cohorts as $addcohort) {
                        $addcohort = trim($addcohort);
                        if (empty($addcohort)) continue;

                        if (is_numeric($addcohort)) {
                            $cohort = $DB->get_record('cohort', ['id' => $addcohort]);
                        } else {
                            $cohort = $DB->get_record('cohort', ['idnumber' => $addcohort]);
                            if (empty($cohort) && has_capability('moodle/cohort:manage', context_system::instance())) {
                                $cohortid = cohort_add_cohort((object)[
                                    'idnumber' => $addcohort,
                                    'name' => $addcohort,
                                    'contextid' => context_system::instance()->id
                                ]);
                                $cohort = $DB->get_record('cohort', ['id' => $cohortid]);
                            }
                        }

                        if (!empty($cohort) && empty($cohort->component)) {
                            if (!$DB->record_exists('cohort_members', ['cohortid' => $cohort->id, 'userid' => $userid])) {
                                cohort_add_member($cohort->id, $userid);
                            }
                        }
                    }

                } catch (Exception $e) {
                    $status = 'Error: ' . $e->getMessage();
                }
            }

            // Collect preview data
            $preview[] = [
                'line' => $key + 1,
                'email' => $username,
                'firstname' => $firstname,
                'lastname' => $lastname,
                'cohorts' => implode(',', $cohorts),
                'status' => $status
            ];
        }

        // Clear draft files
        $fs->delete_area_files($usercontext->id, 'user', 'draft', $draftitemid);

        // Store preview in session for display after redirect
        $SESSION->bulk_upload_preview = $preview;

        // Redirect to same page (PRG pattern)
        redirect(new moodle_url('/local/custompage/bulk_upload.php'));
    }
}

// Display preview table if exists
if (!empty($SESSION->bulk_upload_preview)) {
    echo html_writer::tag('h3', 'Upload Users Preview');
    $table = '<table class="preview-table">';
    $table .= '<tr><th>Line</th><th>Email</th><th>Firstname</th><th>Lastname</th><th>Cohorts</th><th>Status</th></tr>';
    foreach ($SESSION->bulk_upload_preview as $row) {
        $table .= '<tr>';
        $table .= '<td>'.htmlspecialchars($row['line']).'</td>';
        $table .= '<td>'.htmlspecialchars($row['email']).'</td>';
        $table .= '<td>'.htmlspecialchars($row['firstname']).'</td>';
        $table .= '<td>'.htmlspecialchars($row['lastname']).'</td>';
        $table .= '<td>'.htmlspecialchars($row['cohorts']).'</td>';
        $table .= '<td class="'.($row['status'] === 'User created' ? 'notifysuccess' : 'error').'">'.htmlspecialchars($row['status']).'</td>';
        $table .= '</tr>';
    }
    $table .= '</table>';
    echo $table;

    // Clear session after showing
    unset($SESSION->bulk_upload_preview);
}

$form->display();
echo $OUTPUT->footer();
