<?php
require_once(__DIR__ . '/../../config.php');
require_login();

global $DB, $USER, $PAGE, $OUTPUT;

// Page setup
$PAGE->set_url(new moodle_url('/local/custompage/training2.php'));
$PAGE->set_context(context_system::instance());
$PAGE->set_pagelayout('standard');
$PAGE->set_title('Learning Path');
$PAGE->set_heading('Learning Path');

echo $OUTPUT->header();

/* Pagination */
$perpage = 10;
$page = optional_param('page', 0, PARAM_INT);
$start = $page * $perpage;
$today = time();

/* ---------------------------------------------------
 * 1️⃣ Get all PUBLISHED LPs where user is enrolled
 * --------------------------------------------------- */
$sql = "
    SELECT DISTINCT lp.*
      FROM {learning_paths} lp
      JOIN {learning_path_enrolments} e
        ON e.learningpathid = lp.id
     WHERE e.userid = :userid
       AND lp.status = 'Publish'
  ORDER BY lp.id DESC
";

$allpaths = $DB->get_records_sql($sql, ['userid' => $USER->id]);

/* ---------------------------------------------------
 * 2️⃣ FILTER LPs BASED ON COHORT + LTI PROGRESS
 * --------------------------------------------------- */
$filtered = [];

foreach ($allpaths as $lp) {

    /* 🔹 TOTAL courses in LP */
    $totalcourses = $DB->count_records_sql("
        SELECT COUNT(*)
          FROM {learning_path_courses}
         WHERE sectionid IN (
                SELECT id
                  FROM {learning_path_sections}
                 WHERE learningpathid = ?
               )
    ", [$lp->id]);

    /* 🔹 COMPLETED courses (local_ltiprogress) */
    $completedcount = $DB->count_records_sql("
        SELECT COUNT(*)
          FROM {learning_path_courses} lpc
          JOIN {local_ltiprogress} lpgr
            ON lpgr.ltiid = lpc.courseid
           AND lpgr.userid = ?
         WHERE lpc.sectionid IN (
                SELECT id
                  FROM {learning_path_sections}
                 WHERE learningpathid = ?
               )
           AND lpgr.activityprogress = 'Completed'
    ", [$USER->id, $lp->id]);

    /* 🔹 Check cohort / batch expiry */
    $batchexpired = false;

    $enrols = $DB->get_records('learning_path_enrolments', [
        'learningpathid' => $lp->id
    ]);

    foreach ($enrols as $en) {
        if (!empty($en->cohortid)) {
            $cohort = $DB->get_record('cohort', ['id' => $en->cohortid]);
            if ($cohort) {
                if (
                    $cohort->visible == 0 ||
                    (!empty($cohort->end_date) && $cohort->end_date < $today)
                ) {
                    $batchexpired = true;
                    break;
                }
            }
        }
    }

    /* Attach flags */
    $lp->totalcourses   = $totalcourses;
    $lp->completedcount = $completedcount;
    $lp->batchexpired   = $batchexpired;

    $filtered[$lp->id] = $lp;
}

/* ---------------------------------------------------
 * 3️⃣ Pagination after filtering
 * --------------------------------------------------- */
$total = count($filtered);
$paths = array_slice($filtered, $start, $perpage, true);
?>

<style>
:root {
    --primary:#FF4B00;
    --secondary:#F16521;
    --light-bg:#FFF5ED;
    --card-bg:#ffffff;
    --border:#f1e1d8;
    --text-dark:#2b1f1a;
}
.container { max-width:1100px !important; }
h3 { color:var(--primary); font-weight:700; margin-bottom:20px; }

.lp-item {
    background:var(--card-bg);
    padding:18px 20px;
    border:1px solid var(--border);
    border-radius:12px;
    margin-bottom:15px;
    box-shadow:0 2px 8px rgba(0,0,0,0.05);
}
.lp-name { font-size:18px; font-weight:700; color:var(--text-dark); }

.access-btn {
    background: var(--primary);
    color: #fff;
    padding: 8px 18px;
    border-radius: 6px;
    text-decoration: none;
    font-size: 14px;
    font-weight: 600;
    display: inline-flex;
    align-items: center;
    justify-content: center;
    transition: 
        background-color 0.25s ease,
        transform 0.15s ease,
        box-shadow 0.25s ease;
    box-shadow: 0 2px 6px rgba(0,0,0,0.15);
}

/* Hover */
.access-btn:hover {
    background: color-mix(in srgb, var(--primary) 85%, #000);
    color: #fff;
    transform: translateY(-1px);
    box-shadow: 0 6px 14px rgba(0,0,0,0.25);
    text-decoration: none;
}

/* Active (click) */
.access-btn:active {
    transform: translateY(0);
    box-shadow: 0 3px 8px rgba(0,0,0,0.2);
}

/* Keyboard accessibility */
.access-btn:focus-visible {
    outline: none;
    box-shadow: 
        0 0 0 3px rgba(255,255,255,0.8),
        0 0 0 6px color-mix(in srgb, var(--primary) 60%, #fff);
}

.access-disabled {
    background:#ccc; color:#666;
    padding:6px 16px; border-radius:6px;
    font-size:14px; font-weight:600;
    cursor:not-allowed;
}
.toggle-btn {
    background:var(--light-bg);
    border:1px solid var(--secondary);
    color:var(--primary);
    cursor:pointer; border-radius:6px;
    font-size:13px; padding:2px 8px;
}
.course-item { margin:3px 0 3px 30px; font-size:13px; }
.status-msg { margin-top:6px; font-size:13px; font-weight:600; color:#d35400; }
.status-complete { color:#27ae60; }
</style>

<div class="container">
<h3>Learning Path</h3>

<?php foreach ($paths as $lp): ?>

<div class="lp-item">
    <div style="display:flex; justify-content:space-between; align-items:center;">
        <div>
            <div class="lp-name"><?= format_string($lp->name) ?></div>

            <?php if ($lp->batchexpired): ?>
                <div class="status-msg <?= ($lp->completedcount == $lp->totalcourses && $lp->totalcourses > 0) ? 'status-complete' : '' ?>">
                    <?php
                        if ($lp->completedcount == 0) {
                            echo 'Batch expired';
                        } elseif ($lp->completedcount < $lp->totalcourses) {
                            echo 'Status: In progress (Batch expired)';
                        } else {
                            echo 'Status: Completed (Batch expired)';
                        }
                    ?>
                </div>
            <?php endif; ?>
        </div>

        <div style="display:flex; gap:10px;">
            <?php if (!$lp->batchexpired): ?>
                <a class="access-btn"
                   href="<?= new moodle_url('/local/learningpath/learner_path_view.php', ['id'=>$lp->id]) ?>">
                   Access Now
                </a>
            <?php else: ?>
                <span class="access-disabled">Access Disabled</span>
            <?php endif; ?>

            <!-- <button class="toggle-btn"
                onclick="toggleSections(<?= $lp->id ?>)"
                id="btn-lp-<?= $lp->id ?>">▶</button> -->
        </div>
    </div>
</div>

<?php endforeach; ?>

<?= $OUTPUT->paging_bar($total, $page, $perpage,
    new moodle_url('/local/custompage/training2.php')) ?>

</div>

<script>
function toggleSections(id){
    let s = document.getElementById("sections-" + id);
    let b = document.getElementById("btn-lp-" + id);
    if (!s) return; // 👈 prevents error

    if (s.style.display === "none") {
        s.style.display = "block";
        b.innerHTML = "▼";
    } else {
        s.style.display = "none";
        b.innerHTML = "▶";
    }
}

</script>

<?php
echo $OUTPUT->footer();
