<?php
require_once(__DIR__ . '/config.php');
require_login();

/* ================= PERFORMANCE SETTINGS ================= */
set_time_limit(0);
ini_set('memory_limit', '1024M');
@ini_set('output_buffering', 'off');
@ini_set('zlib.output_compression', false);

/* ================= INCLUDES ================= */
require_once($CFG->libdir.'/accesslib.php');
require_once($CFG->dirroot.'/mod/lti/lib.php');
require_once($CFG->dirroot.'/course/lib.php');

global $DB, $PAGE, $OUTPUT;

require_capability('moodle/site:config', context_system::instance());

/* ================= PAGE SETUP ================= */
$PAGE->set_url('/import_full_catalog_modules.php');
$PAGE->set_context(context_system::instance());
$PAGE->set_pagelayout('standard');
$PAGE->set_title('AWS LTI Import');
$PAGE->set_heading('AWS LTI Import – CSV (Preview / Create / Update)');

echo $OUTPUT->header();

/* =====================================================
   FORM
===================================================== */
if (!isset($_POST['submit'])) {
?>
<form method="post" enctype="multipart/form-data" class="mform">
    <div>
        <label><strong>Upload Catalog CSV</strong></label><br>
        <input type="file" name="csvfile" required>
    </div>
    <br>
    <label>
        <input type="checkbox" name="doimport" value="1">
        <strong>Confirm & Import (unchecked = Dry Run)</strong>
    </label>
    <br><br>
    <button type="submit" name="submit" class="btn btn-primary">
        Run
    </button>
</form>
<?php
    echo $OUTPUT->footer();
    exit;
}

/* =====================================================
   CSV PREPARATION
===================================================== */
$file = $_FILES['csvfile']['tmp_name'];
$doimport = !empty($_POST['doimport']);

/* Detect delimiter */
$sample = file_get_contents($file, false, null, 0, 2048);
$delimiter = (substr_count($sample, "\t") > substr_count($sample, ",")) ? "\t" : ",";

$handle = fopen($file, 'r');
$rawheader = fgetcsv($handle, 0, $delimiter);

/* Clean header (BOM + trim) */
$header = [];
foreach ($rawheader as $h) {
    $header[] = trim(preg_replace('/^\xEF\xBB\xBF/', '', $h));
}

/* =====================================================
   GET AWS LTI TOOL ID
===================================================== */
$toolid = $DB->get_field('lti_types', 'id', ['name' => 'AWS'], IGNORE_MULTIPLE);
if (!$toolid) {
    echo $OUTPUT->notification('AWS LTI tool not found in mdl_lti_types', 'error');
    echo $OUTPUT->footer();
    exit;
}

/* =====================================================
   OUTPUT TABLE
===================================================== */
echo "<hr>";
echo $doimport
    ? "<h3>IMPORT MODE (Database will be updated)</h3>"
    : "<h3>DRY RUN MODE (No database changes)</h3>";

echo "<table class='generaltable'>";
echo "<tr>
<th>#</th>
<th>Course ID</th>
<th>Module Name</th>
<th>Status</th>
</tr>";

$rownum = 0;
$created = 0;
$updated = 0;
$skipped = 0;
$errors  = 0;

/* Track courses for cache rebuild */
$affectedcourses = [];

while (($row = fgetcsv($handle, 0, $delimiter)) !== false) {
    $rownum++;

    if (count($row) !== count($header)) {
        $errors++;
        continue;
    }

    $data = array_combine($header, array_map('trim', $row));

    $courseidvalue = $data['course_id'] ?? '';
    $launchurl     = $data['content_launch_url'] ?? '';
    $modulename    = $data['module_name'] ?? 'AWS Module';

    if (!$courseidvalue || !$launchurl) {
        echo "<tr>
                <td>$rownum</td>
                <td>$courseidvalue</td>
                <td>$modulename</td>
                <td style='color:red'>Invalid row</td>
              </tr>";
        $errors++;
        continue;
    }

    /* ---------------- FIND COURSE ---------------- */
    $course = $DB->get_record_sql("
        SELECT c.id
        FROM {course} c
        JOIN {customfield_data} d ON d.instanceid = c.id
        JOIN {customfield_field} f ON f.id = d.fieldid
        WHERE f.shortname = 'course_id'
        AND d.value = ?
    ", [$courseidvalue]);

    if (!$course) {
        echo "<tr>
                <td>$rownum</td>
                <td>$courseidvalue</td>
                <td>$modulename</td>
                <td style='color:red'>Course not found</td>
              </tr>";
        $errors++;
        continue;
    }

    /* ---------------- CHECK EXISTING LTI ---------------- */
    $existinglti = $DB->get_record('lti', [
        'course'  => $course->id,
        'toolurl' => $launchurl
    ]);

    /* =====================================================
       UPDATE EXISTING MODULE NAME
    ===================================================== */
    if ($existinglti) {

        if ($doimport) {
            if ($existinglti->name !== $modulename) {
                $existinglti->name = $modulename;
                $existinglti->timemodified = time();
                $DB->update_record('lti', $existinglti);

                $affectedcourses[$course->id] = true;

                echo "<tr>
                        <td>$rownum</td>
                        <td>$courseidvalue</td>
                        <td>$modulename</td>
                        <td style='color:blue'>Updated module name</td>
                      </tr>";
                $updated++;
            } else {
                echo "<tr>
                        <td>$rownum</td>
                        <td>$courseidvalue</td>
                        <td>$modulename</td>
                        <td style='color:orange'>Duplicate – no change</td>
                      </tr>";
                $skipped++;
            }
        } else {
            echo "<tr>
                    <td>$rownum</td>
                    <td>$courseidvalue</td>
                    <td>$modulename</td>
                    <td style='color:blue'>Would update (dry run)</td>
                  </tr>";
        }

        continue;
    }

    /* =====================================================
       CREATE NEW LTI MODULE
    ===================================================== */
    if ($doimport) {
        try {
            /* Create LTI instance */
            $lti = new stdClass();
            $lti->course      = $course->id;
            $lti->typeid      = $toolid;
            $lti->toolurl     = $launchurl;
            $lti->name        = $modulename;
            $lti->intro       = $data['module_description'] ?? '';
            $lti->introformat = FORMAT_HTML;
            $lti->timecreated = time();
            $lti->timemodified= time();

            $instanceid = lti_add_instance($lti, null);

            if (!$instanceid) {
                throw new Exception('Failed to create LTI instance');
            }

            /* Add to course */
            $module  = $DB->get_record('modules', ['name' => 'lti'], '*', MUST_EXIST);
            $section = $DB->get_record('course_sections', [
                'course'  => $course->id,
                'section' => 0
            ], '*', MUST_EXIST);

            $cm = new stdClass();
            $cm->course       = $course->id;
            $cm->module       = $module->id;
            $cm->instance     = $instanceid;
            $cm->section      = $section->id;
            $cm->visible      = 1;
            $cm->visibleoncoursepage = 1;
            $cm->added        = time();
            $cm->showdescription = 0;

            $cmid = $DB->insert_record('course_modules', $cm);

            /* Update section sequence */
            $section->sequence = $section->sequence
                ? $section->sequence . ',' . $cmid
                : $cmid;

            $DB->update_record('course_sections', $section);

            $affectedcourses[$course->id] = true;

            echo "<tr>
                    <td>$rownum</td>
                    <td>$courseidvalue</td>
                    <td>$modulename</td>
                    <td style='color:green'>Created</td>
                  </tr>";
            $created++;

        } catch (Throwable $e) {
            echo "<tr>
                    <td>$rownum</td>
                    <td>$courseidvalue</td>
                    <td>$modulename</td>
                    <td style='color:red'>{$e->getMessage()}</td>
                  </tr>";
            $errors++;
        }
    } else {
        echo "<tr>
                <td>$rownum</td>
                <td>$courseidvalue</td>
                <td>$modulename</td>
                <td style='color:green'>Ready to create (dry run)</td>
              </tr>";
    }
}

fclose($handle);

/* =====================================================
   REBUILD CACHE (ONCE PER COURSE)
===================================================== */
if ($doimport) {
    foreach (array_keys($affectedcourses) as $cid) {
        rebuild_course_cache($cid, true);
    }
}

/* =====================================================
   SUMMARY
===================================================== */
echo "</table><hr>";
echo "<strong>Summary</strong><br>";
echo "Created: $created<br>";
echo "Updated: $updated<br>";
echo "Skipped: $skipped<br>";
echo "Errors: $errors<br>";

echo $OUTPUT->footer();
