<?php
require('../config.php');
require_login();

$context = context_system::instance();
require_capability('moodle/category:manage', $context);

// Standard Moodle page setup
$PAGE->set_url(new moodle_url('/course/category_display.php'));
$PAGE->set_context($context);
$PAGE->set_pagelayout('report');
$PAGE->set_title(get_string('coursecategories', 'moodle'));
$PAGE->set_heading($SITE->fullname);

echo $OUTPUT->header();

// Load Moodle's flexible table library
require_once($CFG->libdir . '/tablelib.php');

// Add "Add new category" button (top right)
echo html_writer::div(
    $OUTPUT->single_button(
        new moodle_url('/course/editcategory.php', ['parent' => 0]),
        get_string('addnewcategory', 'moodle'),
        'get'
    ),
    'd-flex justify-content-end mb-3'
);

// ----------------------
// HANDLE DELETE ACTION
// ----------------------
$deleteid = optional_param('delete', 0, PARAM_INT);

if ($deleteid) {
    require_sesskey();

    // Count courses in this category
    $coursecount = $DB->count_records('course', ['category' => $deleteid]);

    if ($coursecount > 0) {
        redirect(
            $PAGE->url,
            "You cannot delete this category because it contains {$coursecount} course(s).",
            5,
            \core\output\notification::NOTIFY_WARNING
        );
    }

    // Safe to delete
    $DB->delete_records('course_categories', ['id' => $deleteid]);

    redirect(
        $PAGE->url,
        get_string('categorydeleted', 'moodle'),
        2,
        \core\output\notification::NOTIFY_SUCCESS
    );
}

// Setup flexible table
$table = new flexible_table('course-category-table');

$table->define_columns(['sno', 'uniqueid', 'name', 'courses', 'actions']);
$table->define_headers([
    'S.No',
    'Unique ID',
    get_string('category'),
    get_string('courses'),
    get_string('actions')
]);

$table->define_baseurl($PAGE->url);
$table->sortable(true, 'name', SORT_ASC);
$table->pageable(true);

$totalcount = $DB->count_records('course_categories');
$table->pagesize(10, $totalcount);
$table->set_attribute('class', 'generaltable generalbox boxaligncenter');
$table->setup();

// Get sorting
list($sort, $sortparams) = $table->get_sql_sort();
if (empty($sort)) {
    $sort = 'name ASC';
}

// Fetch paginated data
$categories = $DB->get_records('course_categories', null, $sort,
    '*', $table->get_page_start(), $table->get_page_size());

if ($categories) {
    $rownum = $table->get_page_start() + 1;

    foreach ($categories as $cat) {

        $coursecount = $DB->count_records('course', ['category' => $cat->id]);

        // EDIT icon
        $editurl = new moodle_url('/course/editcategory.php', ['id' => $cat->id]);
        $editicon = $OUTPUT->action_icon(
            $editurl,
            new pix_icon('t/edit', get_string('edit'))
        );

        // DELETE icon (with confirm dialog)
        $deleteurl = new moodle_url('/course/category_display.php', [
            'delete'  => $cat->id,
            'sesskey' => sesskey()
        ]);

        $deleteicon = $OUTPUT->action_icon(
            $deleteurl,
            new pix_icon('t/delete', get_string('delete')),
            new confirm_action('Are you sure you want to delete this category?')
        );

        $actions = $editicon . ' ' . $deleteicon;

        $table->add_data([
            $rownum++,
            $cat->id,
            html_writer::link(
                new moodle_url('/course/index.php', ['categoryid' => $cat->id]),
                format_string($cat->name)
            ),
            $coursecount,
            $actions
        ]);
    }

    $table->finish_output();

} else {
    echo $OUTPUT->notification(get_string('nocategories', 'moodle'), 'notifymessage');
}

echo $OUTPUT->footer();
