<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Cohort related management functions, this file needs to be included manually.
 *
 * @package    core_cohort
 * @copyright  2010 Petr Skoda  {@link http://skodak.org}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require('../config.php');
require_once($CFG->dirroot.'/cohort/lib.php');
require_once($CFG->libdir.'/adminlib.php');

$contextid = optional_param('contextid', 0, PARAM_INT);
$page = optional_param('page', 0, PARAM_INT);
$searchquery  = optional_param('search', '', PARAM_RAW);
$showall = optional_param('showall', false, PARAM_BOOL);


require_login();
// -------------------- DELETE COHORT --------------------
$delete = optional_param('delete', 0, PARAM_INT);
$id     = optional_param('id', 0, PARAM_INT);

if ($delete && $id && confirm_sesskey()) {

    // Load cohort record completely
    $cohort = $DB->get_record('cohort', ['id' => $id], '*', MUST_EXIST);

    // Check permission
    require_capability('moodle/cohort:manage', context::instance_by_id($cohort->contextid));

    // Built-in Moodle delete (also deletes mdl_cohort_members)
    cohort_delete_cohort($cohort);

    // Redirect
    redirect(
        new moodle_url('/cohort/index.php', ['contextid' => $contextid]),
        'Cohort deleted successfully.',
        2
    );
}


if ($contextid) {
    $context = context::instance_by_id($contextid, MUST_EXIST);
} else {
    $context = context_system::instance();
}

if ($context->contextlevel != CONTEXT_COURSECAT and $context->contextlevel != CONTEXT_SYSTEM) {
    print_error('invalidcontext');
}

$category = null;
if ($context->contextlevel == CONTEXT_COURSECAT) {
    $category = $DB->get_record('course_categories', array('id'=>$context->instanceid), '*', MUST_EXIST);
}

$manager = has_capability('moodle/cohort:manage', $context);
$canassign = has_capability('moodle/cohort:assign', $context);
if (!$manager) {
    require_capability('moodle/cohort:view', $context);
}

$strcohorts = get_string('cohorts', 'cohort');

if ($category) {
    $PAGE->set_pagelayout('admin');
    $PAGE->set_context($context);
    $PAGE->set_url('/cohort/index.php', array('contextid'=>$context->id));

    core_course_category::page_setup();
    // Set the cohorts node active in the settings navigation block.
    if ($cohortsnode = $PAGE->settingsnav->find('cohort', navigation_node::TYPE_SETTING)) {
        $cohortsnode->make_active();
    }

    $PAGE->set_title($strcohorts);
    $showall = false;
} else {
    admin_externalpage_setup('cohorts', '', null, '', array('pagelayout'=>'report'));
    $PAGE->set_primary_active_tab('siteadminnode');

      // ✅ Always show "All Cohorts" as active nav item
    $PAGE->navbar->add(get_string('allcohorts', 'cohort'), $PAGE->url);
    $showall = true; // Force all cohorts mode active

    
    // if ($showall == 1) {
    //     $PAGE->navbar->add(get_string('allcohorts', 'cohort'), $PAGE->url);
    //  }// else if (!$showall) {
    // //     $PAGE->navbar->add(get_string('systemcohorts', 'cohort'), $PAGE->url);
    // // }
}

// -------------------- AUTO HIDE / UNHIDE COHORTS BASED ON END DATE --------------------
$now = time();

// 1. Hide expired cohorts (end_date < now)
$expiredcohorts = $DB->get_records_select('cohort',
    'end_date IS NOT NULL AND end_date > 0 AND end_date < :now AND visible = 1',
    ['now' => $now],
    '',
    'id'
);

foreach ($expiredcohorts as $ec) {
    $update = new stdClass();
    $update->id = $ec->id;
    $update->visible = 0;
    $DB->update_record('cohort', $update);
}

// 2. Unhide cohorts when end_date is updated to future (end_date >= now)
$activecohorts = $DB->get_records_select('cohort',
    'end_date IS NOT NULL AND end_date > 0 AND end_date >= :now AND visible = 0',
    ['now' => $now],
    '',
    'id'
);

foreach ($activecohorts as $ac) {
    $update = new stdClass();
    $update->id = $ac->id;
    $update->visible = 1;
    $DB->update_record('cohort', $update);
}
// ------------------------------------------------------------------

echo $OUTPUT->header();

if ($showall) {
    $cohorts = cohort_get_all_cohorts($page, 25, $searchquery);
} else {
    $cohorts = cohort_get_cohorts($context->id, $page, 25, $searchquery);
}

$count = '';
if ($cohorts['allcohorts'] > 0) {
    if ($searchquery === '') {
        $count = ' ('.$cohorts['allcohorts'].')';
    } else {
        $count = ' ('.$cohorts['totalcohorts'].'/'.$cohorts['allcohorts'].')';
    }
}

echo $OUTPUT->heading(get_string('cohortsin', 'cohort', $context->get_context_name()).$count);

$params = array('page' => $page);
if ($contextid) {
    $params['contextid'] = $contextid;
}
if ($searchquery) {
    $params['search'] = $searchquery;
}
if ($showall) {
    $params['showall'] = true;
}
$baseurl = new moodle_url('/cohort/index.php', $params);

if ($editcontrols = cohort_edit_controls($context, $baseurl)) {
    echo $OUTPUT->render($editcontrols);
}

// Add search form.
$hiddenfields = [
    (object) ['name' => 'contextid', 'value' => $contextid],
    (object) ['name' => 'showall', 'value' => $showall]
];

$data = [
    'action' => new moodle_url('/cohort/index.php'),
    'inputname' => 'search',
    'searchstring' => get_string('search', 'cohort'),
    'query' => $searchquery,
    'hiddenfields' => $hiddenfields,
    'extraclasses' => 'mb-3'
];

echo $OUTPUT->render_from_template('core/search_input', $data);

// Output pagination bar.
echo $OUTPUT->paging_bar($cohorts['totalcohorts'], $page, 25, $baseurl);

$data = array();
$editcolumnisempty = true;
$sr = ($page * 25) + 1;   // start counting based on page

foreach ($cohorts['cohorts'] as $cohort) {

    $line = array();

    // Add serial number as first column
    $line[] = $sr++;


    $cohortcontext = context::instance_by_id($cohort->contextid);
    // $cohort->description = file_rewrite_pluginfile_urls($cohort->description, 'pluginfile.php', $cohortcontext->id,
    //         'cohort', 'description', $cohort->id);
    // if ($showall) {
    //     if ($cohortcontext->contextlevel == CONTEXT_COURSECAT) {
    //         $line[] = html_writer::link(new moodle_url('/cohort/index.php' ,
    //                 array('contextid' => $cohort->contextid)), $cohortcontext->get_context_name(false));
    //     } else {
    //         $line[] = $cohortcontext->get_context_name(false);
    //     }
    // }
    $tmpl = new \core_cohort\output\cohortname($cohort);
    $line[] = $OUTPUT->render_from_template('core/inplace_editable', $tmpl->export_for_template($OUTPUT));
    $tmpl = new \core_cohort\output\cohortidnumber($cohort);
    $line[] = $OUTPUT->render_from_template('core/inplace_editable', $tmpl->export_for_template($OUTPUT));
   // $line[] = format_text($cohort->description, $cohort->descriptionformat);

   $usercount = $DB->count_records('cohort_members', ['cohortid' => $cohort->id]);

$line[] = html_writer::link(
    new moodle_url('/cohort/users.php', ['id' => $cohort->id]),
    $usercount,
    ['title' => 'View Users in Batch']
);

//  Source column (component) removed
// if (empty($cohort->component)) {
//     $line[] = get_string('nocomponent', 'cohort');
// } else {
//     $line[] = get_string('pluginname', $cohort->component);
// }
    // ✅ Count courses linked to this cohort
$sql = "SELECT COUNT(DISTINCT c.id)
        FROM {user_enrolments} ue
        JOIN {enrol} e ON e.id = ue.enrolid
        JOIN {course} c ON c.id = e.courseid
        JOIN {cohort_members} cm ON cm.userid = ue.userid
        WHERE cm.cohortid = :cohortid";

$coursecount = $DB->count_records_sql($sql, ['cohortid' => $cohort->id]);

// ✅ Show clickable count
$line[] = html_writer::link(
    new moodle_url('/cohort/wbatch_courses.php', ['id' => $cohort->id]),
    $coursecount > 0 ? $coursecount : '0',
    ['title' => 'View Courses']
);

// fetch company name
$companyname = '';
if (!empty($cohort->company_id)) {
    $company = $DB->get_record('company', ['id' => $cohort->company_id]);
    if ($company) {
        $companyname = $company->com_name;
    }
} else {
    $companyname = '-';
}

$line[] = $companyname; // 👈 Add company name to the table row


    $buttons = array();
if (empty($cohort->component)) {
    $cohortmanager = has_capability('moodle/cohort:manage', $cohortcontext);
    $cohortcanassign = has_capability('moodle/cohort:assign', $cohortcontext);

    $urlparams = array('id' => $cohort->id);

    if ($cohortmanager) {

        // // 👁 Show/Hide Toggle
        // $showhideurl = new moodle_url('/cohort/edit.php', $urlparams + ['sesskey' => sesskey()]);
        // if ($cohort->visible) {
        //     $showhideurl->param('hide', 1);
        //     $buttons[] = html_writer::link($showhideurl,
        //         $OUTPUT->pix_icon('t/hide', get_string('hide')));
        // } else {
        //     $showhideurl->param('show', 1);
        //     $buttons[] = html_writer::link($showhideurl,
        //         $OUTPUT->pix_icon('t/show', get_string('show')));
        // }

        // ❌ Delete – keep default Moodle delete
   $buttons[] = html_writer::link(
    new moodle_url('/cohort/index.php', $urlparams + ['delete' => 1, 'sesskey' => sesskey()]),
    $OUTPUT->pix_icon('t/delete', get_string('delete')),
    ['onclick' => "return confirm('Are you sure you want to delete this cohort?')"]
);


        // ✅ EDIT – redirect to YOUR LOCAL PAGE
        $buttons[] = html_writer::link(
            new moodle_url('/local/companycohort/edit.php', ['id' => $cohort->id]),
            $OUTPUT->pix_icon('t/edit', get_string('edit')),
            ['title' => 'Edit Cohort']
        );

        $editcolumnisempty = false;
    }

    // if ($cohortcanassign) {
    //     $buttons[] = html_writer::link(new moodle_url('/cohort/assign.php', $urlparams),
    //         $OUTPUT->pix_icon('i/users', get_string('assign', 'core_cohort')));
    //     $editcolumnisempty = false;
    // }
}

    $line[] = implode(' ', $buttons);

    $data[] = $row = new html_table_row($line);
    if (!$cohort->visible) {
        $row->attributes['class'] = 'dimmed_text';
    }
}
$table = new html_table();
$table->head  = array(
     'S.No',
    get_string('name', 'cohort'),
    get_string('idnumber', 'cohort'),
    'Users',
    // get_string('component', 'cohort'),
    'Courses',
    'Company' // 👈 added new column
);
$table->colclasses = array('leftalign name', 'leftalign id', 'leftalign description', 'leftalign size','centeralign source');
// if ($showall) {
//     array_unshift($table->head, get_string('category'));
//     array_unshift($table->colclasses, 'leftalign category');
// }
if (!$editcolumnisempty) {
    $table->head[] = get_string('edit');
    $table->colclasses[] = 'centeralign action';
} else {
    // Remove last column from $data.
    foreach ($data as $row) {
        array_pop($row->cells);
    }
}
$table->id = 'cohorts';
$table->attributes['class'] = 'admintable generaltable';
$table->data  = $data;
echo html_writer::table($table);
echo $OUTPUT->paging_bar($cohorts['totalcohorts'], $page, 25, $baseurl);
echo $OUTPUT->footer();
