<?php
// edit.php - cohort editor with user-assign dual-panel
// Place this file in the cohort folder. Assumes edit_form.php defines cohort_edit_form

require('../config.php');
require_once($CFG->dirroot . '/course/lib.php');
require_once($CFG->dirroot . '/cohort/lib.php');
require_once($CFG->dirroot . '/cohort/edit_form.php');

$id        = optional_param('id', 0, PARAM_INT);
$contextid = optional_param('contextid', 0, PARAM_INT);
$delete    = optional_param('delete', 0, PARAM_BOOL);
$show      = optional_param('show', 0, PARAM_BOOL);
$hide      = optional_param('hide', 0, PARAM_BOOL);
$confirm   = optional_param('confirm', 0, PARAM_BOOL);
$returnurl = optional_param('returnurl', '', PARAM_LOCALURL);

require_login();

if ($id) {
    $cohort = $DB->get_record('cohort', ['id' => $id], '*', MUST_EXIST);
    $context = context::instance_by_id($cohort->contextid, MUST_EXIST);

    // Load cohort members for pre-selecting users (comma-separated list)
    $members = $DB->get_records('cohort_members', ['cohortid' => $cohort->id]);
    $cohort->selectedusers = implode(',', array_keys($members));

    // Ensure start/end dates exist (as YYYY-MM-DD or empty string)
    $cohort->start_date = !empty($cohort->start_date) ? $cohort->start_date : '';
    $cohort->end_date   = !empty($cohort->end_date) ? $cohort->end_date : '';

    // Map DB fields to form-friendly names
    $cohort->company_id = isset($cohort->company_id) ? $cohort->company_id : 0;
    $cohort->batchid    = isset($cohort->idnumber) ? $cohort->idnumber : '';

} else {
    // Creating new cohort
    $context = context::instance_by_id($contextid, MUST_EXIST);
    if ($context->contextlevel != CONTEXT_COURSECAT && $context->contextlevel != CONTEXT_SYSTEM) {
        print_error('invalidcontext');
    }

    $cohort = new stdClass();
    $cohort->id            = 0;
    $cohort->contextid     = $context->id;
    $cohort->name          = '';
    $cohort->description   = '';
    $cohort->start_date    = '';
    $cohort->end_date      = '';
    $cohort->company_id    = 0;
    $cohort->batchid       = '';
    $cohort->selectedusers = '';
}

require_capability('moodle/cohort:manage', $context);

$returnurl = $returnurl ? new moodle_url($returnurl) : new moodle_url('/cohort/index.php', ['contextid' => $context->id]);

if (!empty($cohort->component)) {
    // component cohorts cannot be edited here
    redirect($returnurl);
}

$PAGE->set_context($context);
$baseurl = new moodle_url('/cohort/edit.php', ['contextid' => $context->id, 'id' => $cohort->id]);
$PAGE->set_url($baseurl);
$PAGE->set_pagelayout('admin');

$strheading = $cohort->id ? get_string('editcohort', 'cohort') : get_string('addcohort', 'cohort');
$PAGE->set_title($strheading);
$PAGE->navbar->add($strheading);

// Build the form. Pass the cohort as initial data.
$editform = new cohort_edit_form(null, [
    'editoroptions' => [
        'maxfiles' => EDITOR_UNLIMITED_FILES,
        'maxbytes' => $SITE->maxbytes,
        'context'  => $context
    ],
    'data' => $cohort,
    'returnurl' => $returnurl
]);

if ($editform->is_cancelled()) {
    redirect($returnurl);
}

if ($data = $editform->get_data()) {
    // IMPORTANT: Do NOT overwrite form-managed values with optional_param().
    // The correct values are available in $data already.

    // Map form fields to DB columns.
    // idnumber is used for batchid in your DB
    $data->idnumber   = isset($data->batchid) ? trim($data->batchid) : '';

    // company_id is expected in the form as 'company_id' and will already be present in $data.
    // start_date and end_date are expected to be present in $data as provided by the form.

    // Old context (for file moving if context changed)
    $oldcontextid = $context->id;
    // Recompute context from submitted data (if changed)
    $context = context::instance_by_id($data->contextid);
    $editoroptions['context'] = $context;

    if ($data->id) {
        // Editing existing cohort
        if ($data->contextid != $oldcontextid) {
            // Move existing files to new context
            get_file_storage()->move_area_files_to_new_context(
                $oldcontextid, $context->id, 'cohort', 'description', $data->id
            );
        }

        // Process editor (description) and update cohort record
        $data = file_postupdate_standard_editor(
            $data, 'description', $editoroptions, $context, 'cohort', 'description', $data->id
        );

        // Update DB record (this function will call update_record internally)
// Save custom fields into $data (NOT $cohort)
$data->company_id = isset($data->company_id) ? $data->company_id : 0;
$data->start_date = !empty($data->start_date) ? strtotime($data->start_date) : 0;
$data->end_date   = !empty($data->end_date) ? strtotime($data->end_date) : 0;

cohort_update_cohort($data);

        // Reset members and re-add below
        $DB->delete_records('cohort_members', ['cohortid' => $data->id]);

    } else {
        // New cohort
        $data->descriptionformat = $data->description_editor['format'];
        $data->description       = $data->description_editor['text'];

        // Insert cohort and get id
        $data->id = cohort_add_cohort($data);

        // Save files for description
        $data = file_postupdate_standard_editor(
            $data, 'description', $editoroptions, $context, 'cohort', 'description', $data->id
        );

        // If file_postupdate_standard_editor changed the description, update the record
        if ($data->description !== $DB->get_field('cohort', 'description', ['id' => $data->id])) {
            cohort_update_cohort((object)[
                'id' => $data->id,
                'description' => $data->description,
                'contextid' => $context->id
            ]);
        }
    }

    // Save assigned users (selectedusers contains comma separated userids)
    if (!empty($data->selectedusers)) {
        $uids = array_filter(explode(',', $data->selectedusers), function($v) { return trim($v) !== ''; });
        foreach ($uids as $uid) {
            if ($uid) {
                cohort_add_member($data->id, (int)$uid);
            }
        }
    }

    redirect($returnurl);
}

// Render page
echo $OUTPUT->header();

// Minimal CSS for dual-list UI (you can move to a CSS file if desired)
echo '<style>
#fitem_id_description_editor{display:none!important;}
.dual-list{display:flex;gap:10px;margin-top:10px}
.dual-panel{flex:1;border:1px solid #ddd;border-radius:6px;padding:10px;min-height:300px}
.dual-panel .list{max-height:400px;overflow:auto}
.dual-controls{display:flex;flex-direction:column;gap:10px;justify-content:center;padding:10px}
.btn-move{padding:6px 12px}
.user-row{border-bottom:1px solid #f0f0f0;padding:6px}
.user-row label{cursor:pointer}
</style>';

echo $OUTPUT->heading($strheading);

// Display the form
echo $editform->display();

// JS to populate dual-list and handle moving users around. Relies on an ajax endpoint
// that returns JSON with a `users` array (id, fullname, email). Keep this as-is
$js = <<<'JS'
require(['jquery'], function($){
    var selectedusers = $('input[name="selectedusers"]').val();
    selectedusers = selectedusers ? selectedusers.split(',') : [];

    function filterList($input,$list){
        var q=$input.val().toLowerCase();
        $list.find('.user-row').each(function(){
            $(this).toggle($(this).text().toLowerCase().indexOf(q)!==-1);
        });
    }

    function updateHidden(){
        var ids=[];
        $('.selected-users .list .user-row').each(function(){
            ids.push($(this).data('userid'));
        });
        $('input[name="selectedusers"]').val(ids.join(','));
    }

    $.getJSON(M.cfg.wwwroot+'/cohort/ajax_get_users.php',function(resp){
        var htmlAll='', htmlSelected='';
        if(resp && resp.users){
            resp.users.forEach(function(u){
                var userHtml = '<div class="user-row" data-userid="'+u.id+'">'+
                               '<label><input type="checkbox"/> '+u.fullname+'<br><small>'+u.email+'</small></label></div>';
                if(selectedusers.indexOf(u.id.toString())!==-1){
                    htmlSelected += userHtml;
                } else {
                    htmlAll += userHtml;
                }
            });
        }
        $('.all-users .list').html(htmlAll);
        $('.selected-users .list').html(htmlSelected);
    });

    $(document).on('click','.move-right',function(){
        $('.all-users .user-row input:checked').each(function(){
            var $r=$(this).closest('.user-row');$(this).prop('checked',false);
            $('.selected-users .list').append($r);updateHidden();
        });
    });
    $(document).on('click','.move-left',function(){
        $('.selected-users .user-row input:checked').each(function(){
            var $r=$(this).closest('.user-row');$(this).prop('checked',false);
            $('.all-users .list').append($r);updateHidden();
        });
    });
    $(document).on('keyup','.all-users .search input',function(){ filterList($(this),$('.all-users .list')); });
    $(document).on('keyup','.selected-users .search input',function(){ filterList($(this),$('.selected-users .list')); });
    $('form#mform1').on('submit',updateHidden);
});
JS;

$PAGE->requires->js_init_code($js);

echo $OUTPUT->footer();

